/*
 *  Copyright (c) by CryptoSoft GmbH 1998 - 2017
 *  All Rights Reserved
 *  Licensed Material - Property of CryptoSoft GmbH
 *  This software is made available solely pursuant to the
 *  terms of a license agreement which governs its use.
 *
 *  This file encryption sample source will give you an 
 *  idea on how to call up the various library functions.
 *
 *  Usage: cipher [-d] [-a <algo>] [-m <mode>] [-k <key>] [-q] infile outfile
 *    -d        : decrypt
 *    -a <algo> : algorithm (blowfish | cast128 | des3ede | des3eee | des |
                             mars | rijndael | safer | serpent | twofish | cast256)
 *    -m <mode> : operation mode (mode := ecb | cbc | ofb | cfb)
 *    -k <key>  : key used for encryption/decryption
 *    -q        : quiet operation
 *    infile    : name of file to process
 *    outfile   : name of output file
 */

#ifdef WIN32
#include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include "krypt.h"

#define BUFSIZE 16384

#define EMODE_ECB 0   /* electronic code book mode  */
#define EMODE_CBC 1   /* cipher block chaining mode */
#define EMODE_OFB 2   /* 8 bit output feedback mode */
#define EMODE_CFB 3   /* 8 bit cipher feedback mode */

#define EALGO_RIJNDAEL 0  /* Rijndael */
#define EALGO_BLOWFISH 1  /* Blowfish */
#define EALGO_CAST128  2  /* Cast-128 */
#define EALGO_DES      3  /* DES      */
#define EALGO_DES3EDE  4  /* DES3-EDE */
#define EALGO_DES3EEE  5  /* DES3-EEE */
#define EALGO_SAFER    6  /* Safer+   */
#define EALGO_SERPENT  7  /* Serpent  */
#define EALGO_TWOFISH  8  /* Twofish  */
#define EALGO_MARS     9  /* Mars     */
#define EALGO_CAST256  10 /* Cast-256 */

static char *algo_name = "Rijndael"; 

static char *usage = "[-d] [-a <algo> ] [-m <mode>] [-k <key>] [-q] infile outfile";

static BLOWFISH_KS ks_blowfish;
static CAST128_KS  ks_cast128;
static CAST256_KS  ks_cast256;
static DES_KS      ks1_des;
static DES_KS      ks2_des;
static DES_KS      ks3_des;
static RIJNDAEL_KS ks_rijndael;
static SAFER_KS    ks_safer;
static SERPENT_KS  ks_serpent;
static TWOFISH_KS  ks_twofish;
static MARS_KS     ks_mars;

int main(int argc, char **argv)
{
  register unsigned char *buf, *prog, *b;
  static unsigned char key[56]  = {0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF };
  static unsigned char ivec[16] = {0x12, 0x34, 0x56, 0x78, 0x90, 0xAB, 0xCD, 0xEF }, iv[16];
  register int i, bytes, mode, algo, quiet, decrypt, found_problem;
  static int l, ks, bs;
  static unsigned char lb;
  static long fsize;
  register time_t t, t1;
  FILE *ih, *oh;

  quiet = decrypt = found_problem = l = 0;

  algo  = EALGO_RIJNDAEL;  /* default encryption algorithm */
  mode  = EMODE_CBC;       /* default encryption mode */
  ks    = 32;              /* default key size */
  bs    = 16;              /* default block size */

  prog  = *argv;
  argc--, argv++;

  while (argc > 0) {
    if (strcmp(*argv, "-d") == 0) {
      decrypt = 1;
    } else 
    if (strcmp(*argv, "-a") == 0) {
      argc--; argv++;
      if (argc) {
        algo_name = *argv;
        if (strcmp(*argv, "blowfish") == 0) {
          algo = EALGO_BLOWFISH;
          bs = 8; ks = 56;
        } else if (strcmp(*argv, "cast128") == 0) {
          algo = EALGO_CAST128;
          bs = 8; ks = 16;
        } else if (strcmp(*argv, "cast256") == 0) {
          algo = EALGO_CAST256;
          bs = 16; ks = 32;
        } else if (strcmp(*argv, "des3ede") == 0) {
          algo = EALGO_DES3EDE;
          bs = 8; ks = 16;
        } else if (strcmp(*argv, "des3eee") == 0) {
          algo = EALGO_DES3EEE;
          bs = 8; ks = 24;
        } else if (strcmp(*argv, "des3") == 0) {
          algo = EALGO_DES3EDE;
          bs = 8; ks = 16;
        } else if (strcmp(*argv, "des") == 0) {
          algo = EALGO_DES; 
          bs = 8; ks = 8;
        } else if (strcmp(*argv, "mars") == 0) {
          algo = EALGO_MARS; 
          bs = 16; ks = 32;
        } else if (strcmp(*argv, "rijndael") == 0) {
          algo = EALGO_RIJNDAEL;
          bs = 16; ks = 32;
        } else if (strcmp(*argv, "safer") == 0) {
          algo = EALGO_SAFER;
          bs = 16; ks = 32;
        } else if (strcmp(*argv, "serpent") == 0) {
          algo = EALGO_SERPENT;
          bs = 16; ks = 32;
        } else if (strcmp(*argv, "twofish") == 0) {
          algo = EALGO_TWOFISH;
          bs = 16; ks = 32;
        } else found_problem = 1;
      }
    } else
     if (strcmp(*argv, "-m") == 0) {
      argc--; argv++;
      if (argc) {
        if (strcmp(*argv, "ecb") == 0) mode = EMODE_ECB; else
        if (strcmp(*argv, "cbc") == 0) mode = EMODE_CBC; else
        if (strcmp(*argv, "ofb") == 0) mode = EMODE_OFB; else
        if (strcmp(*argv, "cfb") == 0) mode = EMODE_CFB; else
        found_problem = 1;
      }
    } else
    if (strcmp(*argv, "-k") == 0) {
      argc--; argv++;
      if (argc) {
        memset(key, 0, ks);
        bytes = (int) strlen(*argv);
        memcpy(key, *argv, bytes > ks - 1 ? ks:bytes);
      }
    } else
    if (strcmp(*argv, "-q") == 0) {
      quiet = 1;
    } else break;
    argc--, argv++;
  }

  if (argc != 2 || found_problem) {
    fprintf(stderr,"Usage : %s %s \n", prog, usage);
    return 1;
  }

  if ((buf = (unsigned char *) malloc(BUFSIZE)) == NULL) {
    fprintf(stderr,"%s: Error: Cannot alloc memory !\n", prog);
    return 1;
  }

  ih = fopen(*argv++,"rb");  /* open input file for reading */

  oh = fopen(*argv,"wb");    /* create output file */

  if (ih == NULL || oh == NULL) {
    perror(ih ? *argv : *--argv);
    return 1;
  }

  /* get input file size */
  fseek(ih, 0L, SEEK_END);
  fsize = ftell(ih);
  fseek(ih, 0L, SEEK_SET);

  if (decrypt && (mode == EMODE_ECB || mode == EMODE_CBC)) {
    if (fsize % BUFSIZE == 1) found_problem = 1; else found_problem = 0;
    if (found_problem) {
      fseek(ih, fsize - 1, SEEK_SET);
      /* read & save the last byte */
      bytes = (int) fread(&lb, 1, 1, ih);
      fseek(ih, 0L, SEEK_SET);
    }
  }

  /* initialize key schedule array */
  switch(algo) {
    case EALGO_BLOWFISH:
      blowfish_init(key, ks, &ks_blowfish);
      break;
    case EALGO_CAST128:
      cast128_init(key, ks, &ks_cast128);
      break;
    case EALGO_CAST256:
      cast256_init(key, ks, &ks_cast256);
      break;
    case EALGO_DES3EDE:
      des_3EDEinit(key, ks, ks1_des, ks2_des);
      break;
    case EALGO_DES3EEE:
      des_3EEEinit(key, ks, ks1_des, ks2_des, ks3_des);
      break;
    case EALGO_DES:
      des_init(key, ks, ks1_des);
      break;
    case EALGO_MARS:
      mars_init(key, ks, &ks_mars);
      break;
    case EALGO_RIJNDAEL:
      rijndael_init(key, ks, &ks_rijndael);
      break;
    case EALGO_SAFER:
      safer_init(key, ks, &ks_safer);
      break;
    case EALGO_SERPENT:
      serpent_init(key, ks, &ks_serpent);
      break;
    case EALGO_TWOFISH:
      twofish_init(key, ks, &ks_twofish);
      break;
    default:
      break;
  }

  if (mode == EMODE_CBC || mode == EMODE_OFB || mode == EMODE_CFB) memcpy(iv, ivec, bs);

  if (!quiet) t = time(NULL);

  while ((bytes = (int) fread(buf, 1, BUFSIZE, ih)) != 0) {
    if (bytes == 1 && found_problem) break;
    /* block modes */
    if (mode == EMODE_ECB || mode == EMODE_CBC) {
      for (b = buf, i = bytes; bytes > 0; bytes -= bs , b += bs) {
        if (!decrypt) {
          if (bytes < bs) {
            i += bs - bytes;
            l  = bytes;
          }
          switch(algo) {
            case EALGO_BLOWFISH:
              if (mode == EMODE_ECB) {
                blowfish_ecbencode(b, b, &ks_blowfish);
              } else {
                blowfish_cbcencode(b, b, iv, &ks_blowfish);
                memcpy(iv, b, bs);
              }
              break;
            case EALGO_CAST128:
              if (mode == EMODE_ECB) {
                cast128_ecbencode(b, b, &ks_cast128);
              } else {
                cast128_cbcencode(b, b, iv, &ks_cast128);
                memcpy(iv, b, bs);
              }
              break;
            case EALGO_CAST256:
              if (mode == EMODE_ECB) {
                cast256_ecbencode(b, b, &ks_cast256);
              } else {
                cast256_cbcencode(b, b, iv, &ks_cast256);
                memcpy(iv, b, bs);
              }
              break;
            case EALGO_DES3EDE:
              if (mode == EMODE_ECB) {
                des_ecb3EDEencode(b, b, ks1_des, ks2_des);
              } else {
                des_cbc3EDEencode(b, b, iv, ks1_des, ks2_des);
                memcpy(iv, b, bs);
              }
              break;
            case EALGO_DES3EEE:
              if (mode == EMODE_ECB) {
                des_ecb3EEEencode(b, b, ks1_des, ks2_des, ks3_des);
              } else {
                des_cbc3EEEencode(b, b, iv, ks1_des, ks2_des, ks3_des);
                memcpy(iv, b, bs);
              }
              break;
            case EALGO_DES:
              if (mode == EMODE_ECB) {
                des_ecbencode(b, b, ks1_des);
              } else {
                des_cbcencode(b, b, iv, ks1_des);
                memcpy(iv, b, bs);
              }
              break;
            case EALGO_MARS:
              if (mode == EMODE_ECB) {
                mars_ecbencode(b, b, &ks_mars);
              } else {
                mars_cbcencode(b, b, iv, &ks_mars);
                memcpy(iv, b, bs);
              }
              break;
            case EALGO_RIJNDAEL:
              if (mode == EMODE_ECB) {
                rijndael_ecbencode(b, b, &ks_rijndael);
              } else {
                rijndael_cbcencode(b, b, iv, &ks_rijndael);
                memcpy(iv, b, bs);
              }
              break;
            case EALGO_SAFER:
              if (mode == EMODE_ECB) {
                safer_ecbencode(b, b, &ks_safer);
              } else {
                safer_cbcencode(b, b, iv, &ks_safer);
                memcpy(iv, b, bs);
              }
              break;
            case EALGO_SERPENT:
              if (mode == EMODE_ECB) {
                serpent_ecbencode(b, b, &ks_serpent);
              } else {
                serpent_cbcencode(b, b, iv, &ks_serpent);
                memcpy(iv, b, bs);
              }
              break;
            case EALGO_TWOFISH:
              if (mode == EMODE_ECB) {
                twofish_ecbencode(b, b, &ks_twofish);
              } else {
                twofish_cbcencode(b, b, iv, &ks_twofish);
                memcpy(iv, b, bs);
              }
              break;
            default:
              break;
          }
        } else { /* decryption */
          if (bytes > bs - 1) {
            switch(algo) {
              case EALGO_BLOWFISH:
                if (mode == EMODE_ECB) {
                  blowfish_ecbdecode(b, b, &ks_blowfish);
                } else {
                  memcpy(iv, b, bs);
                  blowfish_cbcdecode(b, b, ivec, &ks_blowfish);
                  memcpy(ivec, iv, bs);
                }
                break;
              case EALGO_CAST128:
                if (mode == EMODE_ECB) {
                  cast128_ecbdecode(b, b, &ks_cast128);
                } else {
                  memcpy(iv, b, bs);
                  cast128_cbcdecode(b, b, ivec, &ks_cast128);
                  memcpy(ivec, iv, bs);
                }
                break;
              case EALGO_CAST256:
                if (mode == EMODE_ECB) {
                  cast256_ecbdecode(b, b, &ks_cast256);
                } else {
                  memcpy(iv, b, bs);
                  cast256_cbcdecode(b, b, ivec, &ks_cast256);
                  memcpy(ivec, iv, bs);
                }
		break;
              case EALGO_DES3EDE:
                if (mode == EMODE_ECB) {
                  des_ecb3EDEdecode(b, b, ks1_des, ks2_des);
                } else {
                  memcpy(iv, b, bs);
                  des_cbc3EDEdecode(b, b, ivec, ks1_des, ks2_des);
                  memcpy(ivec, iv, bs);
                }
                break;
              case EALGO_DES3EEE:
                if (mode == EMODE_ECB) {
                  des_ecb3EEEdecode(b, b, ks1_des, ks2_des, ks3_des);
                } else {
                  memcpy(iv, b, bs);
                  des_cbc3EEEdecode(b, b, ivec, ks1_des, ks2_des, ks3_des);
                  memcpy(ivec, iv, bs);
                }
                break;
              case EALGO_DES:
                if (mode == EMODE_ECB) {
                  des_ecbdecode(b, b, ks1_des);
                } else {
                  memcpy(iv, b, bs);
                  des_cbcdecode(b, b, ivec, ks1_des);
                  memcpy(ivec, iv, bs);
                }
                break;
              case EALGO_MARS:
                if (mode == EMODE_ECB) {
                  mars_ecbdecode(b, b, &ks_mars);
                } else {
                  memcpy(iv, b, bs);
                  mars_cbcdecode(b, b, ivec, &ks_mars);
                  memcpy(ivec, iv, bs);
                }
                break;
              case EALGO_RIJNDAEL:
                if (mode == EMODE_ECB) {
                  rijndael_ecbdecode(b, b, &ks_rijndael);
                } else {
                  memcpy(iv, b, bs);
                  rijndael_cbcdecode(b, b, ivec, &ks_rijndael);
                  memcpy(ivec, iv, bs);
                }
                break;
              case EALGO_SAFER:
                if (mode == EMODE_ECB) {
                  safer_ecbdecode(b, b, &ks_safer);
                } else {
                  memcpy(iv, b, bs);
                  safer_cbcdecode(b, b, ivec, &ks_safer);
                  memcpy(ivec, iv, bs);
                }
                break;
              case EALGO_SERPENT:
                if (mode == EMODE_ECB) {
                  serpent_ecbdecode(b, b, &ks_serpent);
                } else {
                  memcpy(iv, b, bs);
                  serpent_cbcdecode(b, b, ivec, &ks_serpent);
                  memcpy(ivec, iv, bs);
                }
                break;
              case EALGO_TWOFISH:
                if (mode == EMODE_ECB) {
                  twofish_ecbdecode(b, b, &ks_twofish);
                } else {
                  memcpy(iv, b, bs);
                  twofish_cbcdecode(b, b, ivec, &ks_twofish);
                  memcpy(ivec, iv, bs);
                }
                break;
              default:
                break;
            }
          } else i -= (bs + 1 - (b[0] & 0xff));
        }
      }
      bytes = i;
      if (decrypt && found_problem && (bytes + 1 == fsize)) bytes = bytes - bs + lb;
    }
    /* 8 bit feedback modes */
    if (mode == EMODE_OFB || mode == EMODE_CFB) {
      if (!decrypt) {
        switch(algo) {
          case EALGO_BLOWFISH:
            if (mode == EMODE_OFB)
            blowfish_ofbencode(buf, buf, bytes, iv, &ks_blowfish);
            else
            blowfish_cfbencode(buf, buf, bytes, iv, &ks_blowfish);
            break;
          case EALGO_CAST128:
            if (mode == EMODE_OFB)
            cast128_ofbencode(buf, buf, bytes, iv, &ks_cast128);
            else
            cast128_cfbencode(buf, buf, bytes, iv, &ks_cast128);
            break;
          case EALGO_CAST256:
            if (mode == EMODE_OFB)
            cast256_ofbencode(buf, buf, bytes, iv, &ks_cast256);
            else
            cast256_cfbencode(buf, buf, bytes, iv, &ks_cast256);
            break;
          case EALGO_DES3EDE:
            if (mode == EMODE_OFB)
            des_ofb3EDEencode(buf, buf, bytes, iv, ks1_des, ks2_des);
            else
            des_cfb3EDEencode(buf, buf, bytes, iv, ks1_des, ks2_des);
            break;
          case EALGO_DES3EEE:
            if (mode == EMODE_OFB)
            des_ofb3EEEencode(buf, buf, bytes, iv, ks1_des, ks2_des, ks3_des);
            else
            des_cfb3EEEencode(buf, buf, bytes, iv, ks1_des, ks2_des, ks3_des);
            break;
          case EALGO_DES:
            if (mode == EMODE_OFB)
            des_ofbencode(buf, buf, bytes, iv, ks1_des);
            else
            des_cfbencode(buf, buf, bytes, iv, ks1_des);
            break;
          case EALGO_MARS:
            if (mode == EMODE_OFB)
            mars_ofbencode(buf, buf, bytes, iv, &ks_mars);
            else
            mars_cfbencode(buf, buf, bytes, iv, &ks_mars);
            break;
          case EALGO_RIJNDAEL:
            if (mode == EMODE_OFB)
            rijndael_ofbencode(buf, buf, bytes, iv, &ks_rijndael);
            else
            rijndael_cfbencode(buf, buf, bytes, iv, &ks_rijndael);
            break;
          case EALGO_SAFER:
            if (mode == EMODE_OFB)
            safer_ofbencode(buf, buf, bytes, iv, &ks_safer);
            else
            safer_cfbencode(buf, buf, bytes, iv, &ks_safer);
            break;
          case EALGO_SERPENT:
            if (mode == EMODE_OFB)
            serpent_ofbencode(buf, buf, bytes, iv, &ks_serpent);
            else
            serpent_cfbencode(buf, buf, bytes, iv, &ks_serpent);
            break;
          case EALGO_TWOFISH:
            if (mode == EMODE_OFB)
            twofish_ofbencode(buf, buf, bytes, iv, &ks_twofish);
            else
            twofish_cfbencode(buf, buf, bytes, iv, &ks_twofish);
            break;
          default:
            break;
        }
      } else { /* decryption */
        switch(algo) {
          case EALGO_BLOWFISH:
            if (mode == EMODE_OFB)
            blowfish_ofbdecode(buf, buf, bytes, iv, &ks_blowfish);
            else
            blowfish_cfbdecode(buf, buf, bytes, iv, &ks_blowfish);
            break;
          case EALGO_CAST128:
            if (mode == EMODE_OFB)
            cast128_ofbdecode(buf, buf, bytes, iv, &ks_cast128);
            else
            cast128_cfbdecode(buf, buf, bytes, iv, &ks_cast128);
            break;
          case EALGO_CAST256:
            if (mode == EMODE_OFB)
            cast256_ofbdecode(buf, buf, bytes, iv, &ks_cast256);
            else
            cast256_cfbdecode(buf, buf, bytes, iv, &ks_cast256);
            break;
          case EALGO_DES3EDE:
            if (mode == EMODE_OFB)
            des_ofb3EDEdecode(buf, buf, bytes, iv, ks1_des, ks2_des);
            else
            des_cfb3EDEdecode(buf, buf, bytes, iv, ks1_des, ks2_des);
            break;
          case EALGO_DES3EEE:
            if (mode == EMODE_OFB)
            des_ofb3EEEdecode(buf, buf, bytes, iv, ks1_des, ks2_des, ks3_des);
            else
            des_cfb3EEEdecode(buf, buf, bytes, iv, ks1_des, ks2_des, ks3_des);
            break;
          case EALGO_DES:
            if (mode == EMODE_OFB)
            des_ofbdecode(buf, buf, bytes, iv, ks1_des);
            else
            des_cfbdecode(buf, buf, bytes, iv, ks1_des);
            break;
          case EALGO_MARS:
            if (mode == EMODE_OFB)
            mars_ofbdecode(buf, buf, bytes, iv, &ks_mars);
            else
            mars_cfbdecode(buf, buf, bytes, iv, &ks_mars);
            break;
          case EALGO_RIJNDAEL:
            if (mode == EMODE_OFB)
            rijndael_ofbdecode(buf, buf, bytes, iv, &ks_rijndael);
            else
            rijndael_cfbdecode(buf, buf, bytes, iv, &ks_rijndael);
            break;
          case EALGO_SAFER:
            if (mode == EMODE_OFB)
            safer_ofbdecode(buf, buf, bytes, iv, &ks_safer);
            else
            safer_cfbdecode(buf, buf, bytes, iv, &ks_safer);
            break;
          case EALGO_SERPENT:
            if (mode == EMODE_OFB)
            serpent_ofbdecode(buf, buf, bytes, iv, &ks_serpent);
            else
            serpent_cfbdecode(buf, buf, bytes, iv, &ks_serpent);
            break;
          case EALGO_TWOFISH:
            if (mode == EMODE_OFB)
            twofish_ofbdecode(buf, buf, bytes, iv, &ks_twofish);
            else
            twofish_cfbdecode(buf, buf, bytes, iv, &ks_twofish);
            break;
          default:
            break;
        }
      }
    }
    if (fwrite(buf, 1, bytes, oh) != (size_t) bytes) {
      fprintf(stderr,"%s: Error in write !\n", prog);
      return 1;
    }
    fsize -= bytes;
  }

  if (!decrypt && mode < EMODE_OFB) fputc(l ? l : l + bs, oh);

  if (!quiet) {
    t1 = time(NULL) - t;
    printf("Processed bytes per second using %s: %ld\n", algo_name, ftell(ih)/(t1 ? t1 : 1));
  }

  fclose(ih);
  fclose(oh);
  return 0;
}
