/*
 *  Copyright (c) by CryptoSoft GmbH 1998-2017
 *  All Rights Reserved
 *  Licensed Material - Property of CryptoSoft GmbH
 *  This software is made available solely pursuant to the
 *  terms of a license agreement which governs its use.
 *
 *  Sample source for ECB mode benchmark
 */

#if defined(WIN16) || defined(WIN32) || defined(WIN64)
#include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <sys/timeb.h>
#include "krypt.h"

void bench(char *algo, fn_ecb pfn_ecb, void *ks, long blksz, long numblk)
{
  register unsigned char *b, *buf;
  register int i, j;
  struct timeb t, t1;
  unsigned long ms;

  if ((buf = (unsigned char *) malloc(blksz * numblk)) == NULL) {
    fprintf(stderr,"Error: Cannot alloc memory !\n");
    return;
  }
  ftime(&t);
  for (j = 0; j < blksz * numblk; j++)
  for (b=buf, i = blksz * numblk; i > 0; i -= blksz , b += blksz) pfn_ecb(b,b,ks);
  ftime(&t1);
  ms = (((long)(t1.time * 1000) + t1.millitm) - ((long)(t.time * 1000) + t.millitm));
  printf("Throughput %10lu bytes/s for %s (ECB mode)\n", (unsigned long)(((float)blksz*numblk*blksz*numblk/ms)) * 1000, algo);
  free(buf);
  return;
}

static char       *algo01 = "16 round Blowfish";
static BLOWFISH_KS algo01_ks;
static char       *algo02 = "16 round Cast-128";
static CAST128_KS  algo02_ks;
static char       *algo03 = "16 round DES     ";
static DES_KS      algo03_ks;
static char       *algo04 = "16 round Rijndael";
static RIJNDAEL_KS algo04_ks;
static char       *algo05 = " 8 round Safer+  ";
static SAFER_KS    algo05_ks;
static char       *algo06 = "32 round Serpent ";
static SERPENT_KS  algo06_ks;
static char       *algo07 = "16 round Twofish ";
static TWOFISH_KS  algo07_ks;
static char       *algo08 = "16 round Mars    ";
static MARS_KS     algo08_ks;
static char       *algo09 = "48 round Cast-256";
static CAST256_KS  algo09_ks;

int main(int argc, char **argv)
{
  bench(algo01, (fn_ecb) &blowfish_ecbencode, &algo01_ks, 8, 1000);
  bench(algo02, (fn_ecb) &cast128_ecbencode,  &algo02_ks, 8, 1000);
  bench(algo03, (fn_ecb) &des_ecbencode,      &(*algo03_ks), 8, 1000);
  bench(algo04, (fn_ecb) &rijndael_ecbencode, &algo04_ks, 16, 500);
  bench(algo05, (fn_ecb) &safer_ecbencode,    &algo05_ks, 16, 500);
  bench(algo06, (fn_ecb) &serpent_ecbencode,  &algo06_ks, 16, 500);
  bench(algo07, (fn_ecb) &twofish_ecbencode,  &algo07_ks, 16, 500);
  bench(algo08, (fn_ecb) &mars_ecbencode,     &algo08_ks, 16, 500);
  bench(algo09, (fn_ecb) &cast256_ecbencode,  &algo09_ks, 16, 500);
  return 0;
}
