#ifndef _MODULE_KRYPT_H_
#define _MODULE_KRYPT_H_
/*
 *             libkrypto - Cryptographic Library
 *
 *    Copyright (c) 1998-2017 by CryptoSoft GmbH. All Rights Reserved.
 *              http://www.cryptosoft.de, info@cryptosoft.de
 *
 */

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#if defined(WIN16)
#define DECL int far pascal
#define UCHAR unsigned char far
#define SIZE_T size_t far
#endif /* WIN16 */

#if defined(OS2) || defined(WIN32)
#define DECL int pascal
#define UCHAR unsigned char
#define SIZE_T size_t
#endif /* OS2 || WIN32 */

#if !defined(WIN16) && !defined(OS2) && !defined(WIN32)
#define DECL int
#define UCHAR unsigned char
#define SIZE_T size_t
#endif /* !WIN16 && !OS2 && !WIN32 */

typedef DECL (*fn_init)(UCHAR *, int, void *);
typedef DECL (*fn_ecb) (UCHAR *, UCHAR *, void *);
typedef DECL (*fn_cbc) (UCHAR *, UCHAR *, UCHAR *, void *);
typedef DECL (*fn_ofb) (UCHAR *, UCHAR *, SIZE_T, UCHAR *, void *);

/*
 *    Synopsis:	algo_init(key,len,pks)
 * Description: intializes all arrays and permutation tables for algo
 *       Input: pks  -  pointer to key schedule array
 *              key  -  key
 *              len  -  length of key
 *      Return: 0 if OK; >0 if a (semi) weak was selected
 *
 *    Synopsis: algo_ecbYcode(in, out, pks)
 * Description: encrypts (Y=en) or decrypts (Y=de) buffer <in> to
 *              buffer <out> using <algo> in ECB mode
 *     	 Input: in  - pointer to buffer of input data
 *              out - pointer to buffer for output data
 *              pks - pointer to key schedule array
 *      Return: 0 if OK
 *
 *    Synopsis: algo_cbcYcode(in, out, iv, pks)
 * Description: encrypts (Y=en) or decrypts (Y=de) buffer <in> to
 *              to buffer <out> using <algo> in CBC mode
 *     	 Input: in  - pointer to buffer of input data
 *              out - pointer to buffer for output data
 *              iv  - pointer to initialization vector
 *              pks - pointer to key schedule array
 *      Output: 0 if OK
 *
 */

/*
 * Algorithm:  Blowfish 
 */

typedef struct blowfish_schedule
{
  unsigned long P[16 + 2];
  unsigned long S[4*256];
} BLOWFISH_KS;

#if defined(WIN16)
#define BLOWFISH_PKS BLOWFISH_KS far *
#endif /* WIN16 */

#if defined(WIN32) || defined(OS2)
#define BLOWFISH_PKS BLOWFISH_KS *
#endif /* WIN32 */

#if !defined(WIN16) && !defined(WIN32) && !defined(OS2)
#define BLOWFISH_PKS BLOWFISH_KS *
#endif /* !WIN16 && !WIN32 && !OS2 */

DECL blowfish_init(UCHAR *key, int len, BLOWFISH_PKS pks);
DECL blowfish_ecbencode(UCHAR *in64, UCHAR *out64, BLOWFISH_PKS pks);
DECL blowfish_ecbdecode(UCHAR *in64, UCHAR *out64, BLOWFISH_PKS pks);
DECL blowfish_cbcencode(UCHAR *in64, UCHAR *out64, UCHAR *iv64, BLOWFISH_PKS pks);
DECL blowfish_cbcdecode(UCHAR *in64, UCHAR *out64, UCHAR *iv64, BLOWFISH_PKS pks);
DECL blowfish_cfbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, BLOWFISH_PKS pks);
DECL blowfish_cfbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, BLOWFISH_PKS pks);
DECL blowfish_ofbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, BLOWFISH_PKS pks);
DECL blowfish_ofbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, BLOWFISH_PKS pks);

/* 
 * Algorithm:   Cast-128
 */

typedef struct cast128_schedule {
  unsigned long ks[32];
} CAST128_KS;

#if defined(WIN16)
#define CAST128_PKS   CAST128_KS far *
#endif /* WIN16 */

#if defined(WIN32) || defined(OS2)
#define CAST128_PKS  CAST128_KS  *
#endif /* WIN32 */

#if !defined(WIN16) && !defined(WIN32) && !defined(OS2)
#define CAST128_PKS  CAST128_KS  *
#endif /* !WIN16 && !WIN32 && !OS2 */

DECL cast128_init(UCHAR *key16, int len, CAST128_PKS pks);
DECL cast128_ecbencode(UCHAR *in64, UCHAR *out64, CAST128_PKS pks);
DECL cast128_ecbdecode(UCHAR *in64, UCHAR *out64, CAST128_PKS pks);
DECL cast128_cbcencode(UCHAR *in64, UCHAR *out64, UCHAR *iv64, CAST128_PKS pks);
DECL cast128_cbcdecode(UCHAR *in64, UCHAR *out64, UCHAR *iv64, CAST128_PKS pks);
DECL cast128_cfbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, CAST128_PKS pks);
DECL cast128_cfbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, CAST128_PKS pks);
DECL cast128_ofbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, CAST128_PKS pks);
DECL cast128_ofbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, CAST128_PKS pks);

/*
 * Algorithm:   DES
 */

#ifdef X64
typedef unsigned int DES_KS[16][2];
#else
typedef unsigned long DES_KS[16][2];
#endif

#if defined(WIN16)
#define DES_PKS       DES_KS     far *
#endif /* WIN16 */

#if defined(WIN32) || defined(OS2)
#define DES_PKS      DES_KS      *
#endif /* WIN32 */

#if !defined(WIN16) && !defined(WIN32) && !defined(OS2)
#define DES_PKS      DES_KS      *
#endif /* !WIN16 && !WIN32 && !OS2 */

DECL des_init(UCHAR *key, int len, DES_KS ks);
DECL des_ecbencode(UCHAR *in64, UCHAR *out64, DES_KS ks);
DECL des_ecbdecode(UCHAR *in64, UCHAR *out64, DES_KS ks);
DECL des_cbcencode(UCHAR *in64, UCHAR *out64, UCHAR *iv64, DES_KS ks);
DECL des_cbcdecode(UCHAR *in64, UCHAR *out64, UCHAR *iv64, DES_KS ks);
DECL des_cfbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, DES_KS ks);
DECL des_cfbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, DES_KS ks);
DECL des_ofbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, DES_KS ks);
DECL des_ofbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, DES_KS ks);
DECL des_3EDEinit(UCHAR *key, int len, DES_KS ks1, DES_KS ks2);
DECL des_ecb3EDEencode(UCHAR *in64, UCHAR *out64, DES_KS ks1, DES_KS ks2);
DECL des_ecb3EDEdecode(UCHAR *in64, UCHAR *out64, DES_KS ks1, DES_KS ks2);
DECL des_cbc3EDEencode(UCHAR *in64, UCHAR *out64, UCHAR *iv64, DES_KS ks1, DES_KS ks2);
DECL des_cbc3EDEdecode(UCHAR *in64, UCHAR *out64, UCHAR *iv64, DES_KS ks1, DES_KS ks2);
DECL des_cfb3EDEencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, DES_KS ks1, DES_KS ks2);
DECL des_cfb3EDEdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, DES_KS ks1, DES_KS ks2);
DECL des_ofb3EDEencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, DES_KS ks1, DES_KS ks2);
DECL des_ofb3EDEdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, DES_KS ks1, DES_KS ks2);
DECL des_3EEEinit(UCHAR *key, int len, DES_KS ks1, DES_KS ks2, DES_KS ks3);
DECL des_ecb3EEEencode(UCHAR *in64, UCHAR *out64, DES_KS ks1, DES_KS ks2, DES_KS ks3);
DECL des_ecb3EEEdecode(UCHAR *in64, UCHAR *out64, DES_KS ks1, DES_KS ks2, DES_KS ks3);
DECL des_cbc3EEEencode(UCHAR *in64, UCHAR *out64, UCHAR *iv64, DES_KS ks1, DES_KS ks2, DES_KS ks3);
DECL des_cbc3EEEdecode(UCHAR *in64, UCHAR *out64, UCHAR *iv64, DES_KS ks1, DES_KS ks2, DES_KS ks3);
DECL des_cfb3EEEencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, DES_KS ks1, DES_KS ks2, DES_KS ks3);
DECL des_cfb3EEEdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, DES_KS ks1, DES_KS ks2, DES_KS ks3);
DECL des_ofb3EEEencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, DES_KS ks1, DES_KS ks2, DES_KS ks3);
DECL des_ofb3EEEdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, DES_KS ks1, DES_KS ks2, DES_KS ks3);

/*
 * Algorihtm:   Rijndael
 */

typedef struct rijndael_schedule {
#ifdef X64
  unsigned int ks[128];
#else
  unsigned long ks[128];
#endif
} RIJNDAEL_KS;

#if defined(WIN16)
#define RIJNDAEL_PKS RIJNDAEL_KS far *
#endif /* WIN16 */

#if defined(WIN32) || defined(OS2)
#define RIJNDAEL_PKS RIJNDAEL_KS *
#endif /* WIN32 */

#if !defined(WIN16) && !defined(WIN32) && !defined(OS2)
#define RIJNDAEL_PKS RIJNDAEL_KS *
#endif /* !WIN16 && !WIN32 && !OS2 */

DECL rijndael_init(UCHAR *key32, int len, RIJNDAEL_PKS pks);
DECL rijndael_ecbencode(UCHAR *in128, UCHAR *out128, RIJNDAEL_PKS pks);
DECL rijndael_ecbdecode(UCHAR *in128, UCHAR *out128, RIJNDAEL_PKS pks);
DECL rijndael_cbcencode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, RIJNDAEL_PKS pks);
DECL rijndael_cbcdecode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, RIJNDAEL_PKS pks);
DECL rijndael_cfbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, RIJNDAEL_PKS pks);
DECL rijndael_cfbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, RIJNDAEL_PKS pks);
DECL rijndael_ofbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, RIJNDAEL_PKS pks);
DECL rijndael_ofbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, RIJNDAEL_PKS pks);

/*
 * Algorithm:   Safer+ 
 */

typedef struct safer_schedule {
  unsigned char ks[529];
} SAFER_KS;

#if defined(WIN16)
#define SAFER_PKS SAFER_KS far *
#endif /* WIN16 */

#if defined(WIN32) || defined(OS2)
#define SAFER_PKS SAFER_KS *
#endif /* WIN32 */

#if !defined(WIN16) && !defined(WIN32) && !defined(OS2)
#define SAFER_PKS SAFER_KS *
#endif /* !WIN16 && !WIN32 && !OS2 */

DECL safer_init(UCHAR *key32, int len, SAFER_PKS pks);
DECL safer_ecbencode(UCHAR *in128, UCHAR *out128, SAFER_PKS pks);
DECL safer_ecbdecode(UCHAR *in128, UCHAR *out128, SAFER_PKS pks);
DECL safer_cbcencode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, SAFER_PKS pks);
DECL safer_cbcdecode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, SAFER_PKS pks);
DECL safer_cfbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, SAFER_PKS pks);
DECL safer_cfbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, SAFER_PKS pks);
DECL safer_ofbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, SAFER_PKS pks);
DECL safer_ofbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, SAFER_PKS pks);

/*
 * Algorithm:   Serpent
 */

typedef struct serpent_schedule {
#ifdef X64
  unsigned int ks[140];
#else
  unsigned long ks[140];
#endif
} SERPENT_KS;

#if defined(WIN16)
#define SERPENT_PKS SERPENT_KS far *
#endif /* WIN16 */

#if defined(WIN32) || defined(OS2)
#define SERPENT_PKS SERPENT_KS *
#endif /* WIN32 */

#if !defined(WIN16) && !defined(WIN32) && !defined(OS2)
#define SERPENT_PKS SERPENT_KS *
#endif /* !WIN16 && !WIN32 && !OS2 */

DECL serpent_init(UCHAR *key32, int len, SERPENT_PKS pks);
DECL serpent_ecbencode(UCHAR *in128, UCHAR *out128, SERPENT_PKS pks);
DECL serpent_ecbdecode(UCHAR *in128, UCHAR *out128, SERPENT_PKS pks);
DECL serpent_cbcencode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, SERPENT_PKS pks);
DECL serpent_cbcdecode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, SERPENT_PKS pks);
DECL serpent_cfbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, SERPENT_PKS pks);
DECL serpent_cfbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, SERPENT_PKS pks);
DECL serpent_ofbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, SERPENT_PKS pks);
DECL serpent_ofbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, SERPENT_PKS pks);

/*
 * Algorithm:   Twofish
 */

typedef struct twofish_schedule {
#ifdef X64
  unsigned int ks[45];
  unsigned int mk[4][256];
#else
  unsigned long ks[45];
  unsigned long mk[4][256];
#endif
} TWOFISH_KS;

#if defined(WIN16)
#define TWOFISH_PKS TWOFISH_KS far *
#endif /* WIN16 */

#if defined(WIN32) || defined(OS2)
#define TWOFISH_PKS TWOFISH_KS *
#endif /* WIN32 */

#if !defined(WIN16) && !defined(WIN32) && !defined(OS2)
#define TWOFISH_PKS TWOFISH_KS *
#endif /* !WIN16 && !WIN32 && !OS2 */

DECL twofish_init(UCHAR *key32, int len, TWOFISH_PKS pks);
DECL twofish_ecbencode(UCHAR *in128, UCHAR *out128, TWOFISH_PKS pks);
DECL twofish_ecbdecode(UCHAR *in128, UCHAR *out128, TWOFISH_PKS pks);
DECL twofish_cbcencode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, TWOFISH_PKS pks);
DECL twofish_cbcdecode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, TWOFISH_PKS pks);
DECL twofish_cfbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, TWOFISH_PKS pks);
DECL twofish_cfbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, TWOFISH_PKS pks);
DECL twofish_ofbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, TWOFISH_PKS pks);
DECL twofish_ofbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, TWOFISH_PKS pks);

/*
 *   Algorithm: Mars
 */

typedef struct mars_schedule {
#ifdef X64
  unsigned int ks[40];
#else
  unsigned long ks[40];
#endif
} MARS_KS;

#if defined(WIN16)
#define MARS_PKS MARS_KS far *
#endif /* WIN16 */

#if defined(WIN32) || defined(OS2)
#define MARS_PKS MARS_KS *
#endif /* WIN32 */

#if !defined(WIN16) && !defined(WIN32) && !defined(OS2)
#define MARS_PKS MARS_KS *
#endif /* !WIN16 && !WIN32 && !OS2 */

DECL mars_init(UCHAR *key32, int len, MARS_PKS pks);
DECL mars_ecbencode(UCHAR *in128, UCHAR *out128, MARS_PKS pks);
DECL mars_ecbdecode(UCHAR *in128, UCHAR *out128, MARS_PKS pks);
DECL mars_cbcencode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, MARS_PKS pks);
DECL mars_cbcdecode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, MARS_PKS pks);
DECL mars_cfbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, MARS_PKS pks);
DECL mars_cfbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, MARS_PKS pks);
DECL mars_ofbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, MARS_PKS pks);
DECL mars_ofbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, MARS_PKS pks);

/*
 *  Algorithm: Cast-256
 */

typedef struct cast256_schedule {
#ifdef X64
  unsigned int ks[96];
#else
  unsigned long ks[96];
#endif
} CAST256_KS;

#if defined(WIN16)
#define CAST256_PKS CAST256_KS far *
#endif /* WIN16 */
 
#if defined(WIN32) || defined(OS2)
#define CAST256_PKS CAST256_KS *
#endif /* WIN32 */                                                              

#if !defined(WIN16) && !defined(WIN32) && !defined(OS2)
#define CAST256_PKS CAST256_KS *
#endif /* !WIN16 && !WIN32 && !OS2 */
                                                                                                        
DECL cast256_init(UCHAR *key32, int len, CAST256_PKS pks);
DECL cast256_ecbencode(UCHAR *in128, UCHAR *out128, CAST256_PKS pks);
DECL cast256_ecbdecode(UCHAR *in128, UCHAR *out128, CAST256_PKS pks);
DECL cast256_cbcencode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, CAST256_PKS pks);
DECL cast256_cbcdecode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, CAST256_PKS pks);
DECL cast256_cfbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, CAST256_PKS pks);
DECL cast256_cfbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, CAST256_PKS pks);
DECL cast256_ofbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, CAST256_PKS pks);
DECL cast256_ofbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, CAST256_PKS pks);

#ifdef __cplusplus
}
#endif /* __cplusplus */
#endif /* _MODULE_KRYPT_H_ */
